//
// (c) 2020 wesolutions GmbH
// All rights reserved.
//

import QtQuick 2.0

import wesual.Controls 1.0
import wesual.data.Core 1.0


PopupAnchorItem {
    id : popup

    property var  items
    property int  activeIndex : -1
    property bool closedOnTab : false

    enabled : visible

    anchors.centerIn : parent

    signal accepted(int result)

    popup : Item {
        id : content

        width  : itemColumn.width  + 2 * bg.border.width
        height : itemColumn.height + 2 * bg.border.width


        anchors {
            top : popup.popupVisible ? parent.bottom : undefined
            left : popup.popupVisible ? parent.left : undefined
            leftMargin : -popup.parent.width / 2
            topMargin : popup.parent.height / 2
        }

        opacity : 0

        states : [
            State {
                name : "visible"
                when : popup.popupVisible

                PropertyChanges {
                    target : content
                    opacity : 1
                }
            }
        ]

        transitions : Transition {
            NumberAnimation {
                property : "opacity"
                duration : 200
            }
        }

        // key handling
        Keys.onEscapePressed : {
            popup.hide();
        }
        Keys.onReturnPressed : {
            popup.accepted(popup.activeIndex >= 0 ?
                           popup.activeIndex : undefined)
        }
        Keys.onEnterPressed : {
            popup.accepted(popup.activeIndex >= 0 ?
                           popup.activeIndex : undefined)
        }
        Keys.onUpPressed : {
            if (items.length < 1)
                return;

            var index = popup.activeIndex - 1;
            if (index < 0)
                index = items.length - 1;
            popup.activeIndex = index;
        }
        Keys.onDownPressed : {
            if (items.length < 1)
                return;

            popup.activeIndex = (popup.activeIndex + 1) % items.length;
        }
        Keys.onTabPressed  : {
            closedOnTab = true;
            popup.hide();

            var next = nextItemInFocusChain(true);
            if (next)
                next.forceActiveFocus();
        }
        Keys.onBacktabPressed : {
            closedOnTab = true;
            popup.hide();
            var next = nextItemInFocusChain(false);
            if (next)
                next.forceActiveFocus();
        }

        Rectangle {
            id : bg

            anchors.fill : parent
            border {
                color : UI.color(UI.PrimaryControlBorder)
                width : 1
            }
        }

        Column {
            id : itemColumn

            readonly property int margin : 8

            readonly property int idealWidth : {
                var max = popup.parent ?
                          (popup.parent.width - bg.border.width * 2) : 0;
                for (var i = 0; i < children.length; ++i) {
                    // popup width should at least be parent.width
                    max = Math.max(max, children[i].implicitWidth,
                                                            popup.parent.width);
                }
                return max;
            }

            x : bg.border.width
            y : bg.border.width
            width : idealWidth
            height : childrenRect.height

            // 3px space before first element
            Item {
                width   : itemColumn.idealWidth
                height  : 3
            }

            Repeater {
                model : popup.items

                delegate : Item {

                    readonly property int itemIndex : index
                    readonly property bool active : popup.activeIndex === index

                    implicitWidth : label.implicitWidth + 2 * itemColumn.margin
                    width : itemColumn.idealWidth
                    height : 26

                    Rectangle {
                        id : itemBackground

                        height : parent.height
                        width : itemColumn.idealWidth
                        color : UI.color(UI.PrimaryBase)

                        states : [
                            State {
                                name : "selected"
                                when : active && mouseArea.pressed

                                PropertyChanges {
                                    target : itemBackground
                                    color : UI.color(UI.PrimaryPress)
                                }
                            },
                            State {
                                name : "hovered"
                                when : active

                                PropertyChanges {
                                    target : itemBackground
                                    color  : UI.color(UI.PrimaryHover)
                                }
                            }
                        ]
                    }

                    Item {
                        id : label

                        implicitWidth : itemText.implicitWidth
                        width : itemColumn.idealWidth
                        height : 26

                        Text {
                            id : itemText
                            text : modelData

                            font {
                                family : UI.fontFamily(UI.PrimaryFont)
                                weight : UI.fontWeight(UI.PrimaryFont)
                                pixelSize : 14
                            }
                            anchors {
                                left : parent.left
                                leftMargin : 10
                                verticalCenter : parent.verticalCenter
                            }
                            color : itemBackground.state === "selected" ?
                                    UI.color(UI.SelectedTextColor) :
                                    UI.color(UI.PrimaryTextColor)
                        }
                    }
                }
            }

            // 3px space after last element
            Item {
                width   : itemColumn.idealWidth
                height  : 3
            }
        }

        MouseArea {
            id : mouseArea
            anchors.fill : parent
            acceptedButtons : Qt.LeftButton
            hoverEnabled : true
            onPositionChanged : {
                var child = itemColumn.childAt(mouse.x, mouse.y);
                if (child && child.itemIndex !== undefined) {
                    popup.activeIndex = child.itemIndex;
                } else {
                    popup.activeIndex = -1;
                }
            }
            onReleased : popup.accepted(popup.activeIndex)
        }
    }
}
